//+------------------------------------------------------------------+
//|	     	                                      WeeklyBreakout.mq5 |
//|                                                    Andriy Moraru |
//|                                         http://www.earnforex.com |
//|            							                            2014 |
//+------------------------------------------------------------------+
#property copyright "www.EarnForex.com, 2014"
#property link      "http://www.earnforex.com"
#property version   "1.01"

#property description "Enters long on previous week's High breakout."
#property description "Enters short on previous week's Low breakout."
#property description "Sets stop-loss on reverse entry."
#property description "Normal exit on Monday, when a new week starts."

#include <Trade/Trade.mqh>
#include <Trade/PositionInfo.mqh>

// Money management
input double Lots = 0.1; 		// Basic lot size.
input bool MM  = false;  	// MM - If true - ATR-based position sizing.
input double Risk = 2; // Risk - Risk tolerance in percentage points.
input double FixedBalance = 0; // FixedBalance - If greater than 0, position size calculator will use it instead of actual account balance.
input double MoneyRisk = 0; // MoneyRisk - Risk tolerance in base currency.
input bool UseMoneyInsteadOfPercentage = false;
input bool UseEquityInsteadOfBalance = false;
input int LotDigits = 2; // LotDigits - How many digits after dot supported in lot size. For example, 2 for 0.01, 1 for 0.1, 3 for 0.001, etc.

// Miscellaneous
input string OrderComment = "WeeklyBreakout";
input int Slippage = 100; 	// Tolerated slippage in brokers' pips.
input bool ECN_Mode = false; // ECN_Mode: Set to true if stop-loss should be added only after Position Open
input int Magic = 132703844;

// Main trading objects
CTrade *Trade;
CPositionInfo PositionInfo;

// Global variables
// Common
ulong LastBars = 0;
bool HaveLongPosition;
bool HaveShortPosition;
double StopLoss; // For MM calculation.
int SecondsPerBar = 0;

MqlRates rates[];
double BuySL, SellSL;

//+------------------------------------------------------------------+
//| Expert Initialization Function                                   |
//+------------------------------------------------------------------+
void OnInit()
{
	// Initialize the Trade class object
	Trade = new CTrade;
	Trade.SetDeviationInPoints(Slippage);
	Trade.SetExpertMagicNumber(Magic);
   SecondsPerBar = PeriodSeconds();
}

//+------------------------------------------------------------------+
//| Expert Deinitialization Function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
	delete Trade;
}

//+------------------------------------------------------------------+
//| Expert Every Tick Function                                       |
//+------------------------------------------------------------------+
void OnTick()
{
   if ((!TerminalInfoInteger(TERMINAL_TRADE_ALLOWED)) || (!TerminalInfoInteger(TERMINAL_CONNECTED)) || (SymbolInfoInteger(_Symbol, SYMBOL_TRADE_MODE) != SYMBOL_TRADE_MODE_FULL)) return;
	
   if (ECN_Mode) DoSL();

	int bars = Bars(_Symbol, _Period);
	
	// Trade only if new bar has arrived
	if (LastBars != bars) LastBars = bars;
	else return;
	
	int copied = CopyRates(NULL, 0, 0, 2, rates); // Two bars are needed - latest completed for OHLC and latest current for Time.
   if (copied <= 0) Print("Error copying price data ", GetLastError());
   
   GetPositionStates();
   
   if ((HaveShortPosition) || (HaveLongPosition)) ClosePrevious();

   double StopLevel = SymbolInfoInteger(_Symbol, SYMBOL_TRADE_STOPS_LEVEL) * _Point;
   double Spread = SymbolInfoInteger(Symbol(), SYMBOL_SPREAD) * _Point;
   
   double Ask = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
   double Bid = SymbolInfoDouble(_Symbol, SYMBOL_BID);

   double UpperEntry = rates[0].high;
   double LowerEntry = rates[0].low;
   
   // Set expiration to the end of the current bar.
   datetime expiration = rates[1].time + SecondsPerBar;

   BuySL = LowerEntry + _Point;
   
   // Current price below entry
   if (UpperEntry - Ask > StopLevel)
   {
      // Skip trade if previous week's range was too tight and does not allow setting normal stop-loss.
      StopLoss = UpperEntry - BuySL;
      if (StopLoss < StopLevel + Spread) return;
      OpenBuyStop(UpperEntry, expiration, BuySL);
   }
   else
   {
      StopLoss = Bid - BuySL;
      if (StopLoss < StopLevel + Spread) return;
      fBuy(BuySL);
   }
   
   SellSL = UpperEntry - _Point;
   
   // Current price above entry
   if (Bid - LowerEntry > StopLevel)
   {
      StopLoss = SellSL - LowerEntry;
      if (StopLoss < StopLevel + Spread) return;
      OpenSellStop(LowerEntry, expiration, SellSL);
   }
   else
   {
      StopLoss = SellSL - Ask;
      if (StopLoss < StopLevel + Spread) return;
      fSell(SellSL);
   }
}

//+------------------------------------------------------------------+
//| Check what position is currently open										|
//+------------------------------------------------------------------+
void GetPositionStates()
{
	// Is there a position on this currency pair?
	if (PositionInfo.Select(_Symbol))
	{
		if (PositionInfo.PositionType() == POSITION_TYPE_BUY)
		{
			HaveLongPosition = true;
			HaveShortPosition = false;
		}
		else if (PositionInfo.PositionType() == POSITION_TYPE_SELL)
		{ 
			HaveLongPosition = false;
			HaveShortPosition = true;
		}
	}
	else 
	{
		HaveLongPosition = false;
		HaveShortPosition = false;
	}
}

//+------------------------------------------------------------------+
//| Buy                                                              |
//+------------------------------------------------------------------+
void fBuy(double SL)
{
	double Ask = SymbolInfoDouble(Symbol(), SYMBOL_ASK);
	if (ECN_Mode) SL = 0;
	Trade.PositionOpen(_Symbol, ORDER_TYPE_BUY, LotsOptimized(), Ask, SL, 0, OrderComment);
}

//+------------------------------------------------------------------+
//| Sell                                                             |
//+------------------------------------------------------------------+
void fSell(double SL)
{
	double Bid = SymbolInfoDouble(Symbol(), SYMBOL_BID);
	if (ECN_Mode) SL = 0;
	Trade.PositionOpen(_Symbol, ORDER_TYPE_SELL, LotsOptimized(), Bid, SL, 0, OrderComment);
}

//+------------------------------------------------------------------+
//| Calculate position size depending on money management parameters.|
//+------------------------------------------------------------------+
double LotsOptimized()
{
	if (!MM) return (Lots);
	
   double Size, RiskMoney, PositionSize = 0;

   // If could not find account currency, probably not connected.
   if (AccountInfoString(ACCOUNT_CURRENCY) == "") return(-1);

   if (FixedBalance > 0)
   {
      Size = FixedBalance;
   }
   else if (UseEquityInsteadOfBalance)
   {
      Size = AccountInfoDouble(ACCOUNT_EQUITY);
   }
   else
   {
      Size = AccountInfoDouble(ACCOUNT_BALANCE);
   }
   
   if (!UseMoneyInsteadOfPercentage) RiskMoney = Size * Risk / 100;
   else RiskMoney = MoneyRisk;

   double UnitCost = SymbolInfoDouble(_Symbol, SYMBOL_TRADE_TICK_VALUE);
   double TickSize = SymbolInfoDouble(_Symbol, SYMBOL_TRADE_TICK_SIZE);
   if ((StopLoss != 0) && (UnitCost != 0) && (TickSize != 0)) PositionSize = NormalizeDouble(RiskMoney / (StopLoss * UnitCost / TickSize), LotDigits);

   if (PositionSize < SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MIN)) PositionSize = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MIN);
   else if (PositionSize > SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MAX)) PositionSize = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MAX);

   return(PositionSize);
} 

//+------------------------------------------------------------------+
//| Close open position																|
//+------------------------------------------------------------------+
void ClosePrevious()
{
	for (int i = 0; i < 10; i++)
	{
		Trade.PositionClose(_Symbol, Slippage);
		if ((Trade.ResultRetcode() != 10008) && (Trade.ResultRetcode() != 10009) && (Trade.ResultRetcode() != 10010))
			Print("Position Close Return Code: ", Trade.ResultRetcodeDescription());
		else return;
	}
}

//+------------------------------------------------------------------+
//| Open Buy Stop Pending Order													|
//+------------------------------------------------------------------+
void OpenBuyStop(double Entry, datetime Expiration, double SL)
{
   // Check if such pending order is already opened
	for (int i = 0; i < OrdersTotal(); i++)
	{
	   OrderGetTicket(i);
	   if ((OrderGetInteger(ORDER_MAGIC) == Magic) && (OrderGetInteger(ORDER_TYPE) == ORDER_TYPE_BUY_STOP)) return;
	}
	
	ulong ticket, tries;
	tries = 0;
	while (tries < 5)
	{
		Trade.OrderOpen(_Symbol, ORDER_TYPE_BUY_STOP, LotsOptimized(), 0, Entry, SL, 0, ORDER_TIME_SPECIFIED, Expiration, OrderComment);
		ticket = Trade.ResultOrder(); // Get ticket
		Print("OpenBuyStop executed. Ticket = " + IntegerToString(ticket) + ".");
		if (ticket <= 0)
		{
			Print("Error occured: " + Trade.ResultRetcodeDescription() + ". BuyStop @ " + DoubleToString(Entry, _Digits) + ".");
			tries++;
		}
		else tries = 5;
	}
}
  
//+------------------------------------------------------------------+
//| Open Sell Stop Pending Order													|
//+------------------------------------------------------------------+
void OpenSellStop(double Entry, datetime Expiration, double SL)
{
   // Check if such pending order is already opened
	for (int i = 0; i < OrdersTotal(); i++)
	{
	   OrderGetTicket(i);
	   if ((OrderGetInteger(ORDER_MAGIC) == Magic) && (OrderGetInteger(ORDER_TYPE) == ORDER_TYPE_SELL_STOP)) return;
	}
	
	ulong ticket, tries;
	tries = 0;
	while (tries < 5)
	{
		Trade.OrderOpen(_Symbol, ORDER_TYPE_SELL_STOP, LotsOptimized(), 0, Entry, SL, 0, ORDER_TIME_SPECIFIED, Expiration, OrderComment);
		ticket = Trade.ResultOrder(); // Get ticket
		Print("OpenSellStop executed. Ticket = " + IntegerToString(ticket) + ".");
		if (ticket <= 0)
		{
			Print("Error occured: " + Trade.ResultRetcodeDescription() + " SellStop @ " + DoubleToString(Entry, _Digits) + ".");
			tries++;
		}
		else tries = 5;
	}
}

//+------------------------------------------------------------------+
//| Applies SL to open positions if ECN mode is on.                  |
//+------------------------------------------------------------------+
void DoSL()
{
   if (!PositionSelect(_Symbol)) return;
   
   double SL = 0;
   
   if (PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY)
   {
      SL = BuySL;
   }
   if (PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_SELL)
   {
      SL = SellSL;
   }

   if ((PositionGetDouble(POSITION_SL) != SL) && (PositionGetDouble(POSITION_SL) == 0))
   {
      Trade.PositionModify(_Symbol, SL, 0);
   }
}
//+------------------------------------------------------------------+