//+------------------------------------------------------------------+
//|                                                    EarnForex.com |
//|                                        https://www.earnforex.com |
//|            							                       2011-2021 |
//+------------------------------------------------------------------+
#property copyright "www.EarnForex.com, 2011-2021"
#property link      "https://www.earnforex.com/metatrader-expert-advisors/ATR-Trailer/"
#property version   "1.05"

#property description "Plain trailing stop EA with ATR-based stop-loss."

#include <Trade/Trade.mqh>
#include <Trade/PositionInfo.mqh>

#define LONG 1
#define SHORT 2

input int ATR_Period = 24;
input int ATR_Multiplier = 3;
input int StartWith = 1; // StartWith: 1 - Short, 2 - Long
input int Slippage = 100; 	// Tolerated slippage in pips
input double Lots = 0.1;
input int TakeProfit = 0; // Take-Profit, in your broker's pips
input bool ECN_Mode = false; // ECN_Mode: Set to true if stop-loss should be added only after Position Open

// Main trading objects
CTrade *Trade;
CPositionInfo PositionInfo;

// Global variables
bool HaveLongPosition;
bool HaveShortPosition;

int LastPosition = 3 - StartWith;

//+------------------------------------------------------------------+
//| Expert Initialization Function                                   |
//+------------------------------------------------------------------+
void OnInit()
{
	// Initialize the Trade class object
	Trade = new CTrade;
	Trade.SetDeviationInPoints(Slippage);
}

//+------------------------------------------------------------------+
//| Expert Deinitialization Function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
	delete Trade;
}

//+------------------------------------------------------------------+
//| Expert Every Tick Function                                       |
//+------------------------------------------------------------------+
void OnTick()
{
	if (TerminalInfoInteger(TERMINAL_TRADE_ALLOWED) == false) return;
	
   // Getting the ATR values
   double ATR[1];
   int myATR = iATR(NULL, 0, ATR_Period);
   if (CopyBuffer(myATR, 0, 0, 1, ATR) != 1) return;
   ATR[0] *= ATR_Multiplier;

   if (ATR[0] <= (SymbolInfoInteger(Symbol(), SYMBOL_TRADE_STOPS_LEVEL) + SymbolInfoInteger(Symbol(), SYMBOL_SPREAD)) * _Point) ATR[0] = (SymbolInfoInteger(Symbol(), SYMBOL_TRADE_STOPS_LEVEL) + SymbolInfoInteger(Symbol(), SYMBOL_SPREAD)) * _Point;

  	// Check what position is currently open
 	GetPositionStates();
   
 	// Adjust SL and TP of the current position
 	if ((HaveLongPosition) || (HaveShortPosition)) AdjustSLTP(ATR[0]);
   else
	{
      double Ask, Bid, SL = 0, TP = 0;
   
   	// Buy condition
   	if (LastPosition == SHORT)
   	{
  			for (int i = 0; i < 10; i++)
  			{
  		   	Ask = SymbolInfoDouble(Symbol(), SYMBOL_ASK);
  				Bid = SymbolInfoDouble(Symbol(), SYMBOL_BID);
  		   	if (!ECN_Mode)
  		   	{
     				// Bid and Ask are swapped to preserve the probabilities and decrease/increase profit/loss size
  		   	   SL = NormalizeDouble(Bid - ATR[0], _Digits);
  		   	   if (TakeProfit) TP = NormalizeDouble(Bid + TakeProfit * _Point, _Digits);
  		   	}
  				Trade.PositionOpen(_Symbol, ORDER_TYPE_BUY, Lots, Ask, SL, TP);
  				Sleep(1000);
  				if ((Trade.ResultRetcode() != 10008) && (Trade.ResultRetcode() != 10009) && (Trade.ResultRetcode() != 10010))
  					Print("Long Position Open Return Code: ", Trade.ResultRetcodeDescription());
  				else return;
  			}
   	}
   	// Sell condition
   	else if (LastPosition == LONG)
   	{
  			for (int i = 0; i < 10; i++)
  			{
  		   	Ask = SymbolInfoDouble(Symbol(), SYMBOL_ASK);
  				Bid = SymbolInfoDouble(Symbol(), SYMBOL_BID);
  		      if (!ECN_Mode)
  		      {
     				// Bid and Ask are swapped to preserve the probabilities and decrease/increase profit/loss size
  		         SL = NormalizeDouble(Ask + ATR[0], _Digits);
  		         if (TakeProfit) TP = NormalizeDouble(Ask - TakeProfit * _Point, _Digits);
  		      }
  				Trade.PositionOpen(_Symbol, ORDER_TYPE_SELL, Lots, Bid, SL, TP);
  				Sleep(1000);
  				if ((Trade.ResultRetcode() != 10008) && (Trade.ResultRetcode() != 10009) && (Trade.ResultRetcode() != 10010))
  					Print("Short Position Open Return Code: ", Trade.ResultRetcodeDescription());
  				else return;
  			}
   	}
	}
}

//+------------------------------------------------------------------+
//| Check What Position is Currently Open										|
//+------------------------------------------------------------------+
void GetPositionStates()
{
	// Is there a position on this currency pair?
	if (PositionInfo.Select(_Symbol))
	{
		if (PositionInfo.PositionType() == POSITION_TYPE_BUY)
		{
			HaveLongPosition = true;
			HaveShortPosition = false;
		}
		else if (PositionInfo.PositionType() == POSITION_TYPE_SELL)
		{ 
			HaveLongPosition = false;
			HaveShortPosition = true;
		}
   	if (HaveLongPosition) LastPosition = LONG;
   	else if (HaveShortPosition) LastPosition = SHORT;
	}
	else 
	{
		HaveLongPosition = false;
		HaveShortPosition = false;
	}
}

//+------------------------------------------------------------------+
//| Adjust Stop-Loss and TakeProfit of the Open Position					|
//+------------------------------------------------------------------+
void AdjustSLTP(double SLparam)
{
	// Is there a position on this currency pair?
	if (PositionInfo.Select(_Symbol))
	{
		if (PositionInfo.PositionType() == POSITION_TYPE_BUY)
		{
		   double Bid = SymbolInfoDouble(Symbol(), SYMBOL_BID);
			double SL = NormalizeDouble(Bid - SLparam, _Digits);
			if ((SL > NormalizeDouble(PositionInfo.StopLoss(), _Digits)) || (NormalizeDouble(PositionInfo.StopLoss(), _Digits) == 0))
			{
				for (int i = 0; i < 10; i++)
				{
					Trade.PositionModify(_Symbol, SL, NormalizeDouble(PositionInfo.TakeProfit(), _Digits));
					if ((Trade.ResultRetcode() != 10008) && (Trade.ResultRetcode() != 10009) && (Trade.ResultRetcode() != 10010))
						Print("Long Position Modify Return Code: ", Trade.ResultRetcodeDescription());
					else return;
				}
			}
		}
		else if (PositionInfo.PositionType() == POSITION_TYPE_SELL)
		{ 
			double Ask = SymbolInfoDouble(Symbol(), SYMBOL_ASK);
			double SL = NormalizeDouble(Ask + SLparam, _Digits);
			if ((SL < NormalizeDouble(PositionInfo.StopLoss(), _Digits)) || (NormalizeDouble(PositionInfo.StopLoss(), _Digits) == 0))
			{
				for (int i = 0; i < 10; i++)
				{
					Trade.PositionModify(_Symbol, SL, NormalizeDouble(PositionInfo.TakeProfit(), _Digits));
					if ((Trade.ResultRetcode() != 10008) && (Trade.ResultRetcode() != 10009) && (Trade.ResultRetcode() != 10010))
						Print("Short Position Modify Return Code: ", Trade.ResultRetcodeDescription());
					else return;
				}
			}
		}
	}
}
//+------------------------------------------------------------------+